package com.nomura.chart.util;

import java.awt.AlphaComposite;
import java.awt.Composite;
import java.awt.Graphics2D;
import java.awt.geom.CubicCurve2D;
import java.awt.geom.Line2D;
import java.awt.geom.QuadCurve2D;
import java.awt.geom.Rectangle2D;
import java.util.Iterator;
import java.util.List;

import org.jfree.chart.plot.PieLabelLinkStyle;
import org.jfree.chart.plot.PieLabelRecord;
import org.jfree.chart.plot.PiePlot;
import org.jfree.chart.plot.PiePlotState;
import org.jfree.chart.plot.PlotRenderingInfo;
import org.jfree.data.DefaultKeyedValues;
import org.jfree.data.KeyedValues;
import org.jfree.data.general.PieDataset;
import org.jfree.text.G2TextMeasurer;
import org.jfree.text.TextBlock;
import org.jfree.text.TextBox;
import org.jfree.text.TextUtilities;
import org.jfree.ui.RectangleAnchor;
import org.jfree.ui.RectangleInsets;

public class ExtendedPiePlot extends PiePlot {
   
    private static final long serialVersionUID = 8511876006195186610L;
    
    public ExtendedPiePlot(){
        super();
    }
    
    public ExtendedPiePlot(PieDataset dataset) {
        super(dataset);
    }
    

    @Override
    protected void drawPie(Graphics2D g2, Rectangle2D plotArea,
            PlotRenderingInfo info) {
        super.drawPie(g2, plotArea, info);
    }

    @Override
    protected void drawLeftLabels(KeyedValues leftKeys, Graphics2D g2,
            Rectangle2D plotArea, Rectangle2D linkArea, float maxLabelWidth,
            PiePlotState state) {
        this.getLabelDistributor().clear();
        double lGap = plotArea.getWidth() * this.getLabelGap();
        double verticalLinkRadius = state.getLinkArea().getHeight() / 2.0;
        for (int i = 0; i < leftKeys.getItemCount(); i++) {
            String label = this.getLabelGenerator().generateSectionLabel(
                    this.getDataset(), leftKeys.getKey(i));
            if (label != null) {
                TextBlock block = TextUtilities.createTextBlock(label,
                        this.getLabelFont(), this.getLabelPaint(), 200,
                        new G2TextMeasurer(g2));
                TextBox labelBox = new TextBox(block);
                labelBox.setBackgroundPaint(this.getLabelBackgroundPaint());
                labelBox.setOutlinePaint(this.getLabelOutlinePaint());
                labelBox.setOutlineStroke(this.getLabelOutlineStroke());
                labelBox.setShadowPaint(this.getLabelShadowPaint());
                labelBox.setInteriorGap(this.getLabelPadding());
                double theta = Math.toRadians(
                        leftKeys.getValue(i).doubleValue());
                double baseY = state.getPieCenterY() - Math.sin(theta)
                               * verticalLinkRadius;
                double hh = labelBox.getHeight(g2);

                this.getLabelDistributor().addPieLabelRecord(new PieLabelRecord(
                        leftKeys.getKey(i), theta, baseY, labelBox, hh,
                        lGap / 2.0 + lGap / 2.0 * -Math.cos(theta), 1.0
                        - getLabelLinkDepth()
                        + getExplodePercent(leftKeys.getKey(i))));
            }
        }
        double hh = plotArea.getHeight();
        double gap = hh * getInteriorGap();
        this.getLabelDistributor().distributeLabels(plotArea.getMinY() + gap,
                hh - 2 * gap);
        for (int i = 0; i < this.getLabelDistributor().getItemCount(); i++) {
            drawLeftLabel(g2, state,
                    this.getLabelDistributor().getPieLabelRecord(i));
        }
    }

    @Override
    protected void drawRightLabels(KeyedValues keys, Graphics2D g2,
            Rectangle2D plotArea, Rectangle2D linkArea, float maxLabelWidth,
            PiePlotState state) {
     // draw the right labels...
        this.getLabelDistributor().clear();
        double lGap = plotArea.getWidth() * this.getLabelGap();
        double verticalLinkRadius = state.getLinkArea().getHeight() / 2.0;

        for (int i = 0; i < keys.getItemCount(); i++) {
            String label = this.getLabelGenerator().generateSectionLabel(
                    this.getDataset(), keys.getKey(i));

            if (label != null) {
                TextBlock block = TextUtilities.createTextBlock(label,
                        this.getLabelFont(), this.getLabelPaint(), 200,
                        new G2TextMeasurer(g2));
                TextBox labelBox = new TextBox(block);
                labelBox.setBackgroundPaint(this.getLabelBackgroundPaint());
                labelBox.setOutlinePaint(this.getLabelOutlinePaint());
                labelBox.setOutlineStroke(this.getLabelOutlineStroke());
                labelBox.setShadowPaint(this.getLabelShadowPaint());
                labelBox.setInteriorGap(this.getLabelPadding());
                double theta = Math.toRadians(keys.getValue(i).doubleValue());
                double baseY = state.getPieCenterY()
                              - Math.sin(theta) * verticalLinkRadius;
                double hh = labelBox.getHeight(g2);
                this.getLabelDistributor().addPieLabelRecord(new PieLabelRecord(
                        keys.getKey(i), theta, baseY, labelBox, hh,
                        lGap / 2.0 + lGap / 2.0 * Math.cos(theta),
                        1.0 - getLabelLinkDepth()
                        + getExplodePercent(keys.getKey(i))));
            }
        }
        double hh = plotArea.getHeight();
        double gap = hh * getInteriorGap();
        this.getLabelDistributor().distributeLabels(plotArea.getMinY() + gap,
                hh - 2 * gap);
        for (int i = 0; i < this.getLabelDistributor().getItemCount(); i++) {
            drawRightLabel(g2, state,
                    this.getLabelDistributor().getPieLabelRecord(i));
        }

    }

    @Override
    protected void drawLeftLabel(Graphics2D g2, PiePlotState state,
            PieLabelRecord record) {
        double anchorX = state.getLinkArea().getMinX() - 129;
        double targetX = anchorX;
        double targetY = record.getAllocatedY();        

        if ( getLabelLinksVisible() ) {
            double theta = record.getAngle();
            double linkX = state.getPieCenterX() + Math.cos(theta)
                    * state.getPieWRadius() * record.getLinkPercent();
            double linkY = state.getPieCenterY() - Math.sin(theta)
                    * state.getPieHRadius() * record.getLinkPercent();
            double elbowX = state.getPieCenterX() + Math.cos(theta)
                    * state.getLinkArea().getWidth() / 2.0;
            double elbowY = state.getPieCenterY() - Math.sin(theta)
                    * state.getLinkArea().getHeight() / 2.0;
            double anchorY = elbowY;
            
            g2.setPaint( getLabelLinkPaint() );
            g2.setStroke( getLabelLinkStroke() );
            PieLabelLinkStyle style = getLabelLinkStyle();
            if (style.equals(PieLabelLinkStyle.STANDARD)) {
                g2.draw(new Line2D.Double(linkX, linkY, elbowX, elbowY));
                g2.draw(new Line2D.Double(anchorX, anchorY, elbowX, elbowY));
                g2.draw(new Line2D.Double(anchorX, anchorY, targetX, targetY));
            }
            else if (style.equals(PieLabelLinkStyle.QUAD_CURVE)) {
                QuadCurve2D q = new QuadCurve2D.Float();
                q.setCurve(targetX, targetY, anchorX, anchorY, elbowX, elbowY);
                g2.draw(q);
                g2.draw(new Line2D.Double(elbowX, elbowY, linkX, linkY));
            }
            else if (style.equals(PieLabelLinkStyle.CUBIC_CURVE)) {
                CubicCurve2D c = new CubicCurve2D .Float();
                c.setCurve(targetX, targetY, anchorX, anchorY, elbowX, elbowY,
                        linkX, linkY);
                g2.draw(c);
            }
        }
        TextBox tb = record.getLabel();
        tb.setShadowXOffset(0D);
        tb.setShadowXOffset(0D); 
        tb.setInteriorGap(new RectangleInsets(0D, 2D, 0D, 0D));
        tb.draw(g2, (float) targetX - 2 , (float) targetY + 8, RectangleAnchor.LEFT);
    }

    @Override
    protected void drawRightLabel(Graphics2D g2, PiePlotState state,
            PieLabelRecord record) {
        double anchorX = state.getLinkArea().getMaxX() + 128;
        double targetX = anchorX ;
        double targetY = record.getAllocatedY();

        if (this.getLabelLinksVisible()) {
            double theta = record.getAngle();
            double linkX = state.getPieCenterX() + Math.cos(theta)
                    * state.getPieWRadius() * record.getLinkPercent();
            double linkY = state.getPieCenterY() - Math.sin(theta)
                    * state.getPieHRadius() * record.getLinkPercent();
            double elbowX = state.getPieCenterX() + Math.cos(theta)
                    * state.getLinkArea().getWidth() / 2.0;
            double elbowY = state.getPieCenterY() - Math.sin(theta)
                    * state.getLinkArea().getHeight() / 2.0 ;
            double anchorY = elbowY;
            g2.setPaint(this.getLabelLinkPaint());
            g2.setStroke(this.getLabelLinkStroke());
            PieLabelLinkStyle style = getLabelLinkStyle();
            if (style.equals(PieLabelLinkStyle.STANDARD)) {
                g2.draw(new Line2D.Double(linkX, linkY, elbowX, elbowY));
                g2.draw(new Line2D.Double(anchorX, anchorY, elbowX, elbowY));
                g2.draw(new Line2D.Double(anchorX, anchorY, targetX, targetY));
            }
            else if (style.equals(PieLabelLinkStyle.QUAD_CURVE)) {
                QuadCurve2D q = new QuadCurve2D.Float();
                q.setCurve(targetX, targetY, anchorX, anchorY, elbowX, elbowY);
                g2.draw(q);
                g2.draw(new Line2D.Double(elbowX, elbowY, linkX, linkY));
            }
            else if (style.equals(PieLabelLinkStyle.CUBIC_CURVE)) {
                CubicCurve2D c = new CubicCurve2D .Float();
                c.setCurve(targetX, targetY, anchorX, anchorY, elbowX, elbowY,
                        linkX, linkY);
                g2.draw(c);
            }
        }
        TextBox tb = record.getLabel();
        tb.setShadowXOffset(0D);
        tb.setShadowXOffset(0D);        
        tb.setInteriorGap(new RectangleInsets(0D, 0D, 0D, 0D));
        tb.draw(g2, (float) targetX , (float) targetY + 8, RectangleAnchor.RIGHT);
    }

    @Override
    protected void drawLabels(Graphics2D g2, List keys, double totalValue,
            Rectangle2D plotArea, Rectangle2D linkArea, PiePlotState state) {
        Composite originalComposite = g2.getComposite();
        g2.setComposite(AlphaComposite.getInstance(AlphaComposite.SRC_OVER,
                1.0f));

        // classify the keys according to which side the label will appear...
        DefaultKeyedValues leftKeys = new DefaultKeyedValues();
        DefaultKeyedValues rightKeys = new DefaultKeyedValues();

        double runningTotal = 0.0;
        Iterator iterator = keys.iterator();
        while (iterator.hasNext()) {
            Comparable key = (Comparable) iterator.next();
            boolean include = true;
            double v = 0.0;
            Number n = this.getDataset().getValue(key);
            if (n == null) {
                include = !this.getIgnoreNullValues();
            }
            else {
                v = n.doubleValue();
                include = this.getIgnoreZeroValues() ? v > 0.0 : v >= 0.0;
            }

            if (include) {
                runningTotal = runningTotal + v;
                // work out the mid angle (0 - 90 and 270 - 360) = right,
                // otherwise left
                double mid = this.getStartAngle() + (this.getDirection().getFactor()
                        * ((runningTotal - v / 2.0) * 360) / totalValue);
                if (Math.cos(Math.toRadians(mid)) < 0.0) {
                    leftKeys.addValue(key, new Double(mid));
                }
                else {
                    rightKeys.addValue(key, new Double(mid));
                }
            }
        }

        g2.setFont(getLabelFont());

        // calculate the max label width from the plot dimensions, because
        // a circular pie can leave a lot more room for labels...
        double marginX= plotArea.getX() + this.getInteriorGap()
                * plotArea.getWidth();
        double gap = plotArea.getWidth() * this.getLabelGap();
        double ww = linkArea.getX() - gap - marginX;
        float labelWidth = (float) this.getLabelPadding().trimWidth(ww);

        // draw the labels...
        if (this.getLabelGenerator() != null) {
            drawLeftLabels(leftKeys, g2, plotArea, linkArea, labelWidth,
                    state);
            drawRightLabels(rightKeys, g2, plotArea, linkArea, labelWidth,
                    state);
        }
        g2.setComposite(originalComposite);

    }  
    
    

}
